import { db } from "@db";
import { eq } from "drizzle-orm";
import { 
  services, 
  serviceFeatures, 
  testimonials, 
  contactInquiries,
  type InsertContactInquiry,
  type Service,
  type ServiceFeature,
  type Testimonial,
  type ContactInquiry
} from "@shared/schema";

export const storage = {
  // Services
  async getServices(): Promise<(Service & { features: string[] })[]> {
    try {
      const allServices = await db.query.services.findMany({
        with: {
          features: true
        }
      });
      
      // Transform the result to include features as string array
      return allServices.map(service => ({
        ...service,
        features: service.features.map(feature => feature.feature)
      }));
    } catch (error) {
      console.error("Error fetching services:", error);
      throw error;
    }
  },

  async getServiceBySlug(slug: string): Promise<(Service & { features: string[] }) | null> {
    try {
      const service = await db.query.services.findFirst({
        where: eq(services.slug, slug),
        with: {
          features: true
        }
      });
      
      if (!service) {
        return null;
      }
      
      return {
        ...service,
        features: service.features.map(feature => feature.feature)
      };
    } catch (error) {
      console.error(`Error fetching service by slug ${slug}:`, error);
      throw error;
    }
  },

  // Testimonials
  async getTestimonials(): Promise<Testimonial[]> {
    try {
      return await db.query.testimonials.findMany();
    } catch (error) {
      console.error("Error fetching testimonials:", error);
      throw error;
    }
  },

  // Contact inquiries
  async createContactInquiry(data: InsertContactInquiry): Promise<ContactInquiry> {
    try {
      const [inquiry] = await db.insert(contactInquiries)
        .values(data)
        .returning();
        
      return inquiry;
    } catch (error) {
      console.error("Error creating contact inquiry:", error);
      throw error;
    }
  },

  async getContactInquiries(): Promise<ContactInquiry[]> {
    try {
      return await db.query.contactInquiries.findMany();
    } catch (error) {
      console.error("Error fetching contact inquiries:", error);
      throw error;
    }
  }
};
