import { pgTable, text, serial, integer, boolean, timestamp } from "drizzle-orm/pg-core";
import { createInsertSchema } from "drizzle-zod";
import { z } from "zod";
import { relations } from "drizzle-orm";

export const users = pgTable("users", {
  id: serial("id").primaryKey(),
  username: text("username").notNull().unique(),
  password: text("password").notNull(),
});

export const insertUserSchema = createInsertSchema(users).pick({
  username: true,
  password: true,
});

export type InsertUser = z.infer<typeof insertUserSchema>;
export type User = typeof users.$inferSelect;

// Contact form inquiries
export const contactInquiries = pgTable("contact_inquiries", {
  id: serial("id").primaryKey(),
  fullName: text("full_name").notNull(),
  email: text("email").notNull(),
  company: text("company").notNull(),
  serviceInterest: text("service_interest").notNull(),
  message: text("message").notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  isContacted: boolean("is_contacted").default(false).notNull(),
});

export const contactInquirySchema = createInsertSchema(contactInquiries, {
  fullName: (schema) => schema.min(2, "Name must be at least 2 characters"),
  email: (schema) => schema.email("Must provide a valid email"),
  company: (schema) => schema.min(1, "Company name is required"),
  serviceInterest: (schema) => schema.min(1, "Please select a service interest"),
  message: (schema) => schema.min(10, "Message must be at least 10 characters"),
});

export type ContactInquiry = typeof contactInquiries.$inferSelect;
export type InsertContactInquiry = z.infer<typeof contactInquirySchema>;

// Testimonials
export const testimonials = pgTable("testimonials", {
  id: serial("id").primaryKey(),
  name: text("name").notNull(),
  position: text("position").notNull(),
  company: text("company").notNull(),
  rating: integer("rating").notNull(),
  testimonial: text("testimonial").notNull(),
  initials: text("initials").notNull(),
  bgColor: text("bg_color").notNull(),
  textColor: text("text_color").notNull(),
});

export const testimonialsSchema = createInsertSchema(testimonials);

export type Testimonial = typeof testimonials.$inferSelect;
export type InsertTestimonial = z.infer<typeof testimonialsSchema>;

// Services
export const services = pgTable("services", {
  id: serial("id").primaryKey(),
  name: text("name").notNull(),
  slug: text("slug").notNull().unique(),
  shortDescription: text("short_description").notNull(),
  iconClass: text("icon_class").notNull(),
  colorClass: text("color_class").notNull(),
  borderClass: text("border_class").notNull(),
});

export const serviceFeatures = pgTable("service_features", {
  id: serial("id").primaryKey(),
  serviceId: integer("service_id").references(() => services.id).notNull(),
  feature: text("feature").notNull(),
});

export const servicesRelations = relations(services, ({ many }) => ({
  features: many(serviceFeatures),
}));

export const serviceFeaturesRelations = relations(serviceFeatures, ({ one }) => ({
  service: one(services, { fields: [serviceFeatures.serviceId], references: [services.id] }),
}));

export const servicesSchema = createInsertSchema(services);
export const serviceFeaturesSchema = createInsertSchema(serviceFeatures);

export type Service = typeof services.$inferSelect;
export type InsertService = z.infer<typeof servicesSchema>;
export type ServiceFeature = typeof serviceFeatures.$inferSelect;
export type InsertServiceFeature = z.infer<typeof serviceFeaturesSchema>;
